package gov.va.med.mhv.vitals.validator;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.util.Date;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.vitals.dto.BodyWeightReadingDTO;
import gov.va.med.mhv.vitals.enums.WeightMeasureEnumeration;
import gov.va.med.mhv.vitals.util.MessagesUtil;

@Component
public class BodyWeightValidator extends Validator {
	
	private static final BigDecimal BODYWEIGHT_MIN_VALUE = new BigDecimal("10");
	private static final BigDecimal BODYWEIGHT_MAX_VALUE = new BigDecimal("999.99");
	private static final int BODYWEIGHT_SCALE = 2;
	public static final int COMMENTS_MAX_LENGTH = 255;
	
	@Autowired
	private MessagesUtil messages;
	
	public void bodyWeightValidations(ResponseUtil response, BodyWeightReadingDTO bwdto) {
		fieldValidation(response, bwdto);
	}

	private void fieldValidation(ResponseUtil response,	BodyWeightReadingDTO bwdto) {
		validateBodyWeight(response,bwdto);
		validateComments(response,bwdto);
		validateMeasure(response,bwdto);
		validateReadingDate(response,bwdto);
		validateUserProfile(response,bwdto);
	}
	
	private void validateUserProfile(ResponseUtil response,	BodyWeightReadingDTO bwdto) {
		if( bwdto.getUserprofileId() == null ) {
			response.getValidationErrors().put("userProfile", "userProfile is required");
			return;
		}
		boolean valid= isUserProfileValid(bwdto.getUserprofileId());
		if ( !valid) {
			response.getValidationErrors().put("userProfile", "userProfile is Invalid");
		}
		
	}

	private void validateReadingDate(ResponseUtil response,	BodyWeightReadingDTO bwdto) {
		Date value = bwdto.getReading();
		if( value == null ) {
			Validations.validateRequired("readingDate", value, response.getValidationErrors());
			return;
		}
		
	}

	private void validateMeasure(ResponseUtil response,	BodyWeightReadingDTO bwdto) {
		String value=bwdto.getBodyweightMeasure();
		if ( value == null ) {
			Validations.validateRequired("measure", value, response.getValidationErrors());
			return;
		}	
		Validations.validateEnumeration("measure", value, WeightMeasureEnumeration.values(), response.getValidationErrors());
	}

	private void validateComments(ResponseUtil response, BodyWeightReadingDTO bwdto) {
		String value = bwdto.getComments();
		if ( value == null ) {
			return;
		}						
		Validations.validateMaxLength("comments",value, COMMENTS_MAX_LENGTH, response.getValidationErrors());
	}

	private void validateBodyWeight(ResponseUtil response, BodyWeightReadingDTO bwdto) {
		BigDecimal value = bwdto.getBodyweight();
		if( value == null ) {
			Validations.validateRequired("bodyWeight", value, response.getValidationErrors());
			return;
		}		
		Validations.validateMinValue("Body Weight minumum", value, BODYWEIGHT_MIN_VALUE,"The Body Weight ", response.getValidationErrors());
		Validations.validateMaxValue("Body Weight maximum",value, BODYWEIGHT_MAX_VALUE,"The Body Weight ", response.getValidationErrors());
		Validations.validateScale("Body Weight scale",value, BODYWEIGHT_SCALE, "The Body Weight ", response.getValidationErrors());
	}
	
	public void addError(ResponseUtil response, String errorKey,String value, Object[]  args) {
		response.getValidationErrors().put(errorKey,MessageFormat.format(value, args));
	}  


}
